// src/controllers/userController.js
import { differenceInCalendarDays } from 'date-fns';
import User from '../models/User.js';
import calcRenewalDatee from '../utils/calculateDate.js';
import { calcRenewalDate } from '../utils/util.js';
import sendTelegramNotification from '../utils/sendTelegram.js';

export const getUsers = async (req, res) => {
  const { filterType } = req.query;
  const users = await User.find().lean().exec();
  const now = new Date();

  const enriched = users.map(u => {
    const renewalDate = u.renewalDate ?? calcRenewalDate(u.datePayment, u.plan);
    const daysUntilRenewal = Math.floor((renewalDate - now) / (1000 * 60 * 60 * 24));
    return { ...u, renewalDate, daysUntilRenewal };
  });

  let filtered = enriched;
  if (filterType === 'expire') filtered = enriched.filter(u => u.daysUntilRenewal <= 3);
  if (filterType === 'review') filtered = enriched.filter(u => u.daysUntilRenewal > 3 && u.daysUntilRenewal <= 30);
  if (filterType === 'new') {
    filtered = enriched.filter(u => {
      const joined = new Date(u.dateOfJoin);
      return u.daysUntilRenewal > 30 && (now - joined)/(1000*60*60*24) <= 90;
    });
  }

  res.json(filtered);
};



export const addUser = async (req, res) => {
  try {
    const data = req.body;
    console.log('Adding user:', data);

    const plan = data.plan.toLowerCase();
    
    const renewalDate = calcRenewalDate(data.startPlanData, plan);

    const newUser = new User({
      ...data,
      plan,
      renewalDate,
    });

    await newUser.save();

    console.log('User added successfully');
    res.status(201).json(newUser);
  } catch (error) {
    console.error('Error adding user:', error);
    res.status(500).json({ message: 'Internal Server Error', error: error.message });
  }
};





export const updateUser = async (req, res) => {
  const { id } = req.params;
  const data = req.body;

  const renewalDate = calcRenewalDate(new Date(data.datePayment), data.plan.toLowerCase());

  try {
    const updated = await User.findByIdAndUpdate(
      id,
      { ...data, renewalDate },
      { new: true }
    ).exec();

    if (updated) {
      res.json(updated);
    } else {
      res.status(404).json({ message: 'User not found' });
    }
  } catch (err) {
    res.status(500).json({ message: 'Server error', error: err.message });
  }
};


export const deleteUser = async (req, res) => {
  const { id } = req.params;
  try {
    const deleted = await User.findByIdAndDelete(id).exec();
    if (deleted) {
      res.status(200).json({ message: 'User deleted successfully' });
    } else {
      res.status(404).json({ message: 'User not found' });
    }
  } catch (error) {
    res.status(500).json({ message: 'Server error', error: error.message });
  }
};




export const getDashboardUsers = async (req, res) => {
  try {
    const allUsers = await User.find().lean();
    const today = new Date();

    const usersWithRenewalInfo = allUsers.map(user => {
      const planEndDate = calcRenewalDatee(user.startPlanData, user.plan);
      const daysUntilRenewal = differenceInCalendarDays(planEndDate, today);

      return {
        _id: user._id,
        name: user.name,
        telegramId: user.telegramId,
        dateOfJoin: user.dateOfJoin,
        datePayment: user.datePayment,
        amount: parseFloat(user.amount) || 0,
        plan: user.plan,
        planEndDate,
        daysUntilRenewal
      };
    });

    res.status(200).json(usersWithRenewalInfo);
  } catch (err) {
    console.error('Dashboard fetch error:', err);
    res.status(500).json({ message: 'Dashboard fetch failed', error: err.message });
  }
};




import dotenv from 'dotenv';
import axios from 'axios';

dotenv.config();

const BOT_TOKEN = process.env.TELEGRAM_BOT_TOKEN;

export const renewUser = async (req, res) => {
  const { id } = req.params;

  console.log('🔄 Renewing user with Telegram ID:', id);

  try {
    const user = await User.findOne({ telegramId: id });
    if (!user) return res.status(404).json({ message: 'User not found' });

    const newRenewalDate = calcRenewalDate(new Date(), user.plan);

    user.datePayment = new Date();
    user.renewalDate = newRenewalDate;
    await user.save();

    const message = `✅ Hi *${user.name}*, your subscription has been renewed until *${newRenewalDate.toDateString()}*. Thank you!`;

    const response = await axios.post(`https://api.telegram.org/bot${BOT_TOKEN}/sendMessage`, {
      chat_id: user.telegramId,
      text: message,
      parse_mode: 'Markdown',
    });

    console.log('✅ Telegram message sent:', response.data);

    res.status(200).json({ message: 'Renewal successful', user });
  } catch (err) {
    console.error('❌ Renewal error:', err?.response?.data || err.message);
    res.status(500).json({ message: 'Renewal failed', error: err.message });
  }
};


